<?php
session_start();
require_once 'config.php';
require_once 'sheets_api.php';

$error = '';
$message = '';
$isLoggedIn = isset($_SESSION['user_id']);
$userData = $isLoggedIn ? [
    'id' => $_SESSION['user_id'],
    'email' => $_SESSION['user_email'],
    'name' => $_SESSION['user_name'],
    'role' => $_SESSION['user_role']
] : null;

// Get all users from Google Sheets for display
$allUsers = [];
$spreadsheetId = defined('SPREADSHEET_ID') && !empty(SPREADSHEET_ID) ? SPREADSHEET_ID : '';

if (!empty($spreadsheetId)) {
    try {
        // Get all data from the "users" sheet
        $sheetName = 'users';
        $sheetData = getSheetData($spreadsheetId, $sheetName);
        
        if (!empty($sheetData) && count($sheetData) > 1) {
            $headers = $sheetData[0];
            
            // Get column indexes
            $emailIndex = array_search('email', $headers);
            $passwordIndex = array_search('password', $headers);
            $idIndex = array_search('id', $headers);
            $nameIndex = array_search('name', $headers);
            $roleIndex = array_search('role', $headers);
            $statusIndex = array_search('status', $headers);
            
            // Check if required columns exist
            if ($emailIndex !== false && $passwordIndex !== false) {
                // Get all users
                for ($i = 1; $i < count($sheetData); $i++) {
                    $row = $sheetData[$i];
                    
                    // Check if the row has enough columns
                    if (count($row) <= max($emailIndex, $passwordIndex)) {
                        continue;
                    }
                    
                    // Add user to the list
                    $allUsers[] = [
                        'id' => $idIndex !== false && isset($row[$idIndex]) ? $row[$idIndex] : $i,
                        'email' => $row[$emailIndex],
                        'password' => $row[$passwordIndex],
                        'name' => $nameIndex !== false && isset($row[$nameIndex]) ? $row[$nameIndex] : 'User',
                        'role' => $roleIndex !== false && isset($row[$roleIndex]) ? $row[$roleIndex] : 'user',
                        'status' => $statusIndex !== false && isset($row[$statusIndex]) ? $row[$statusIndex] : 'active'
                    ];
                }
            }
        }
    } catch (Exception $e) {
        $error = "Error loading users: " . $e->getMessage();
    }
}

// Handle login
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['login'])) {
    $email = isset($_POST['email']) ? trim($_POST['email']) : '';
    $password = isset($_POST['password']) ? $_POST['password'] : '';
    
    if (empty($email) || empty($password)) {
        $error = "Please enter both email and password";
    } else {
        try {
            if (empty($spreadsheetId)) {
                $error = "Spreadsheet ID is not configured. Please set it in config.php";
            } else {
                // Get user data from the "users" sheet
                $sheetName = 'users';
                $userData = authenticateUser($spreadsheetId, $sheetName, $email, $password);
                
                if ($userData) {
                    // Set session variables
                    $_SESSION['user_id'] = $userData['id'];
                    $_SESSION['user_email'] = $userData['email'];
                    $_SESSION['user_name'] = $userData['name'];
                    $_SESSION['user_role'] = $userData['role'];
                    
                    // Set logged in status
                    $isLoggedIn = true;
                    
                    // Redirect to avoid form resubmission
                    header('Location: ' . $_SERVER['PHP_SELF'] . '?login_success=1');
                    exit;
                } else {
                    $error = "Invalid email or password";
                }
            }
        } catch (Exception $e) {
            $error = "Error: " . $e->getMessage();
        }
    }
}

// Handle success message after redirect
if (isset($_GET['login_success'])) {
    $message = "Login successful! Welcome, " . $_SESSION['user_name'];
}

// Handle logout
if (isset($_GET['logout'])) {
    session_destroy();
    $_SESSION = [];
    
    // Redirect to remove the logout parameter
    header("Location: " . $_SERVER['PHP_SELF']);
    exit;
}

// Get user-specific data from Google Sheets
$userSpecificData = [];

if ($isLoggedIn && !empty($spreadsheetId)) {
    try {
        // Get all data from the "user_data" sheet
        $sheetName = 'user_data';
        $allData = getSheetData($spreadsheetId, $sheetName);
        
        if (!empty($allData)) {
            // Filter data for the specific user
            $headers = $allData[0];
            $userIdIndex = array_search('user_id', $headers);
            
            if ($userIdIndex !== false) {
                for ($i = 1; $i < count($allData); $i++) {
                    // Compare as strings to handle both numeric and string IDs
                    if (isset($allData[$i][$userIdIndex]) &&
                        (string)$allData[$i][$userIdIndex] === (string)$_SESSION['user_id']) {
                        $row = [];
                        foreach ($allData[$i] as $index => $cell) {
                            if (isset($headers[$index])) {
                                $row[$headers[$index]] = $cell;
                            }
                        }
                        $userSpecificData[] = $row;
                    }
                }
            }
        }
    } catch (Exception $e) {
        $error = "Error retrieving user data: " . $e->getMessage();
    }
}

/**
 * Authenticate user against Google Sheets data
 * 
 * @param string $spreadsheetId The ID of the spreadsheet
 * @param string $sheetName The name of the sheet containing user data
 * @param string $email The user's email
 * @param string $password The user's password
 * @return array|false User data if authenticated, false otherwise
 */
function authenticateUser($spreadsheetId, $sheetName, $email, $password) {
    try {
        // Get all data from the sheet
        $allData = getSheetData($spreadsheetId, $sheetName);
        
        if (empty($allData) || count($allData) < 2) {
            // No data or only headers
            return false;
        }
        
        // Get column indexes
        $headers = $allData[0];
        $emailIndex = array_search('email', $headers);
        $passwordIndex = array_search('password', $headers);
        $idIndex = array_search('id', $headers);
        $nameIndex = array_search('name', $headers);
        $roleIndex = array_search('role', $headers);
        $statusIndex = array_search('status', $headers);
        
        // Check if required columns exist
        if ($emailIndex === false || $passwordIndex === false) {
            throw new Exception("Required columns 'email' and 'password' not found in the sheet");
        }
        
        // Search for matching email and password
        for ($i = 1; $i < count($allData); $i++) {
            $row = $allData[$i];
            
            // Check if the row has enough columns
            if (count($row) <= max($emailIndex, $passwordIndex)) {
                continue;
            }
            
            // Check if email and password match
            if ($row[$emailIndex] === $email && $row[$passwordIndex] === $password) {
                // Check if account is active (if status column exists)
                if ($statusIndex !== false && isset($row[$statusIndex]) && $row[$statusIndex] !== 'active') {
                    return false; // Account is not active
                }
                
                // Return user data
                return [
                    'id' => $idIndex !== false && isset($row[$idIndex]) ? $row[$idIndex] : $i,
                    'email' => $row[$emailIndex],
                    'name' => $nameIndex !== false && isset($row[$nameIndex]) ? $row[$nameIndex] : 'User',
                    'role' => $roleIndex !== false && isset($row[$roleIndex]) ? $row[$roleIndex] : 'user'
                ];
            }
        }
        
        // No matching user found
        return false;
    } catch (Exception $e) {
        throw $e;
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Real Google Sheets Login</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0-alpha1/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        body {
            padding: 20px;
            background-color: #f8f9fa;
        }
        .container {
            max-width: 1000px;
        }
        .card {
            margin-bottom: 20px;
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
        }
        .login-form {
            max-width: 400px;
            margin: 0 auto;
        }
        .user-info {
            background-color: #e9ecef;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
        }
        .table-responsive {
            margin-top: 20px;
        }
        .users-table th, .users-table td {
            vertical-align: middle;
        }
        .status-active {
            color: green;
            font-weight: bold;
        }
        .status-inactive {
            color: red;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1 class="my-4 text-center">Real Google Sheets Login</h1>
        
        <?php if (!empty($error)): ?>
        <div class="alert alert-danger"><?php echo htmlspecialchars($error); ?></div>
        <?php endif; ?>
        
        <?php if (!empty($message)): ?>
        <div class="alert alert-success"><?php echo htmlspecialchars($message); ?></div>
        <?php endif; ?>
        
        <?php if ($isLoggedIn): ?>
            <!-- User is logged in, show dashboard -->
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5 class="mb-0">User Dashboard</h5>
                    <a href="?logout=1" class="btn btn-sm btn-outline-danger">Logout</a>
                </div>
                <div class="card-body">
                    <div class="user-info">
                        <h4>Welcome, <?php echo htmlspecialchars($userData['name']); ?>!</h4>
                        <p><strong>Email:</strong> <?php echo htmlspecialchars($userData['email']); ?></p>
                        <p><strong>Role:</strong> <?php echo htmlspecialchars($userData['role']); ?></p>
                        <p><strong>User ID:</strong> <?php echo htmlspecialchars($userData['id']); ?></p>
                    </div>
                    
                    <h5>Your Personal Data</h5>
                    
                    <?php if (empty($userSpecificData)): ?>
                        <div class="alert alert-info">
                            No personal data found for your user ID. Make sure your Google Sheet has:
                            <ul>
                                <li>A sheet named 'user_data'</li>
                                <li>A column named 'user_id'</li>
                                <li>Rows with your user ID (<?php echo htmlspecialchars($userData['id']); ?>)</li>
                            </ul>
                        </div>
                    <?php else: ?>
                        <div class="table-responsive">
                            <table class="table table-striped table-bordered">
                                <thead>
                                    <tr>
                                        <?php foreach (array_keys($userSpecificData[0]) as $header): ?>
                                            <?php if ($header !== 'user_id'): ?>
                                                <th><?php echo htmlspecialchars($header); ?></th>
                                            <?php endif; ?>
                                        <?php endforeach; ?>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($userSpecificData as $row): ?>
                                        <tr>
                                            <?php foreach ($row as $key => $value): ?>
                                                <?php if ($key !== 'user_id'): ?>
                                                    <td><?php echo htmlspecialchars($value); ?></td>
                                                <?php endif; ?>
                                            <?php endforeach; ?>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        <?php else: ?>
            <!-- User is not logged in, show login form and users table -->
            <div class="row">
                <div class="col-md-6">
                    <div class="card">
                        <div class="card-header">
                            <h5 class="mb-0">Login with Google Sheets</h5>
                        </div>
                        <div class="card-body">
                            <form method="post" action="" class="login-form">
                                <div class="mb-3">
                                    <label for="email" class="form-label">Email</label>
                                    <input type="email" class="form-control" id="email" name="email" required>
                                </div>
                                <div class="mb-3">
                                    <label for="password" class="form-label">Password</label>
                                    <input type="password" class="form-control" id="password" name="password" required>
                                </div>
                                <button type="submit" name="login" class="btn btn-primary">Login</button>
                            </form>
                        </div>
                    </div>
                </div>
                
                <div class="col-md-6">
                    <div class="card">
                        <div class="card-header">
                            <h5 class="mb-0">Available Users for Testing</h5>
                        </div>
                        <div class="card-body">
                            <?php if (empty($allUsers)): ?>
                                <div class="alert alert-warning">
                                    No users found in the Google Sheet. Make sure your Google Sheet has:
                                    <ul>
                                        <li>A sheet named 'users'</li>
                                        <li>Columns: id, email, password, name, role, status</li>
                                        <li>At least one user row</li>
                                    </ul>
                                </div>
                            <?php else: ?>
                                <div class="table-responsive">
                                    <table class="table table-sm table-bordered users-table">
                                        <thead>
                                            <tr>
                                                <th>ID</th>
                                                <th>Email</th>
                                                <th>Password</th>
                                                <th>Name</th>
                                                <th>Role</th>
                                                <th>Status</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($allUsers as $user): ?>
                                                <tr>
                                                    <td><?php echo htmlspecialchars($user['id']); ?></td>
                                                    <td><?php echo htmlspecialchars($user['email']); ?></td>
                                                    <td><?php echo htmlspecialchars($user['password']); ?></td>
                                                    <td><?php echo htmlspecialchars($user['name']); ?></td>
                                                    <td><?php echo htmlspecialchars($user['role']); ?></td>
                                                    <td class="<?php echo $user['status'] === 'active' ? 'status-active' : 'status-inactive'; ?>">
                                                        <?php echo htmlspecialchars($user['status']); ?>
                                                    </td>
                                                </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        <?php endif; ?>
        
        <div class="card mt-4">
            <div class="card-header">
                <h5 class="mb-0">About This Page</h5>
            </div>
            <div class="card-body">
                <p>This page demonstrates a real login system using Google Sheets as a user database:</p>
                <ul>
                    <li>Displays all users from your Google Sheet for testing</li>
                    <li>Authenticates users against the actual data in your Google Sheet</li>
                    <li>Shows user-specific data after successful login</li>
                </ul>
                
                <div class="alert alert-info">
                    <strong>Note:</strong> Make sure your Google Spreadsheet has:
                    <ol>
                        <li>A sheet named 'users' with columns: id, email, password, name, role, status</li>
                        <li>A sheet named 'user_data' with columns: user_id, data_field_1, data_field_2, etc.</li>
                    </ol>
                </div>
                
                <a href="index.php" class="btn btn-primary">Back to Main Page</a>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0-alpha1/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>