<?php
// First, check if setup is complete
$setupComplete = false;
$setupMessage = '';

// Include config.php to check if it has valid credentials
require_once 'config.php';

// Check if config.php has valid credentials
if (defined('CLIENT_ID') && !empty(CLIENT_ID) && CLIENT_ID !== 'your-client-id' &&
    defined('CLIENT_EMAIL') && !empty(CLIENT_EMAIL) && CLIENT_EMAIL !== 'your-service-account-email' &&
    defined('PRIVATE_KEY') && !empty(PRIVATE_KEY) && PRIVATE_KEY !== 'your-private-key') {
    
    $setupComplete = true;
} else {
    $setupMessage = 'Configuration is incomplete. Please complete the setup process.';
}

// If setup is not complete, redirect to setup.php
if (!$setupComplete) {
    // For AJAX requests, return JSON
    if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest') {
        header('Content-Type: application/json');
        echo json_encode(['setup_complete' => false, 'message' => $setupMessage]);
        exit;
    }
    
    // For regular requests, redirect to setup.php
    header('Location: setup.php');
    exit;
}

// If we get here, setup is complete, so continue with the main application

require_once 'sheets_api.php';

$spreadsheetId = '';
$sheetName = '';
$data = [];
$sheetNames = [];
$error = '';

// Check if form is submitted
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Get spreadsheet ID from form
    if (isset($_POST['spreadsheet_id']) && !empty($_POST['spreadsheet_id'])) {
        $spreadsheetId = $_POST['spreadsheet_id'];
        $selectedSheet = isset($_POST['sheet_name']) ? $_POST['sheet_name'] : '';
        
        // Save spreadsheet ID to config
        $configContent = file_get_contents('config.php');
        $newConfigContent = preg_replace(
            "/define\('SPREADSHEET_ID', '(.*)'\);/",
            "define('SPREADSHEET_ID', '$spreadsheetId');",
            $configContent
        );
        file_put_contents('config.php', $newConfigContent);
        
        // Redirect to the same page with parameters to avoid form resubmission
        $redirectUrl = 'index.php?id=' . urlencode($spreadsheetId);
        if (!empty($selectedSheet)) {
            $redirectUrl .= '&sheet=' . urlencode($selectedSheet);
        }
        header('Location: ' . $redirectUrl);
        exit;
    } else {
        $error = 'Please enter a Spreadsheet ID';
    }
}

// Process GET parameters (after redirect)
if (isset($_GET['id']) && !empty($_GET['id'])) {
    $spreadsheetId = $_GET['id'];
    
    try {
        // Get sheet names
        $sheetNames = getSheetNames($spreadsheetId);
        
        // Get selected sheet or use the first one
        if (isset($_GET['sheet']) && !empty($_GET['sheet'])) {
            $sheetName = $_GET['sheet'];
        } else if (!empty($sheetNames)) {
            $sheetName = $sheetNames[0];
        }
        
        // Get data from the selected sheet
        if (!empty($sheetName)) {
            $range = urlencode($sheetName);
            $data = getSheetData($spreadsheetId, $range);
        }
    } catch (Exception $e) {
        $error = $e->getMessage();
    }
}

// Use spreadsheet ID from config if not provided
if (empty($spreadsheetId) && defined('SPREADSHEET_ID') && !empty(SPREADSHEET_ID)) {
    $spreadsheetId = SPREADSHEET_ID;
    
    try {
        // Get sheet names
        $sheetNames = getSheetNames($spreadsheetId);
        
        // Get selected sheet or use the first one
        if (empty($sheetName) && !empty($sheetNames)) {
            $sheetName = $sheetNames[0];
        }
        
        // Get data from the selected sheet
        if (!empty($sheetName)) {
            $range = urlencode($sheetName);
            $data = getSheetData($spreadsheetId, $range);
        }
    } catch (Exception $e) {
        $error = $e->getMessage();
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Google Sheets Database Demo</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0-alpha1/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        body {
            padding: 20px;
        }
        .container {
            max-width: 1200px;
        }
        .table-responsive {
            margin-top: 20px;
        }
        .error {
            color: red;
            margin-top: 10px;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1 class="mb-4">Google Sheets Database Demo</h1>
        
        <div class="card mb-4">
            <div class="card-header">
                <h5>Connect to Google Sheet</h5>
            </div>
            <div class="card-body">
                <form method="post" action="index.php">
                    <div class="mb-3">
                        <label for="spreadsheet_id" class="form-label">Spreadsheet ID</label>
                        <input type="text" class="form-control" id="spreadsheet_id" name="spreadsheet_id"
                               value="<?php echo htmlspecialchars($spreadsheetId); ?>"
                               placeholder="Enter the ID from your Google Sheet URL">
                        <div class="form-text">
                            The Spreadsheet ID is the value between /d/ and /edit in the Google Sheet URL.<br>
                            Example: https://docs.google.com/spreadsheets/d/<strong>1BxiMVs0XRA5nFMdKvBdBZjgmUUqptlbs74OgvE2upms</strong>/edit
                        </div>
                    </div>
                    
                    <?php if (!empty($sheetNames)): ?>
                    <div class="mb-3">
                        <label for="sheet_name" class="form-label">Sheet Name</label>
                        <select class="form-select" id="sheet_name" name="sheet_name">
                            <?php foreach ($sheetNames as $sheet): ?>
                            <option value="<?php echo htmlspecialchars($sheet); ?>"
                                    <?php echo ($sheet === $sheetName) ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($sheet); ?>
                            </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <?php endif; ?>
                    
                    <button type="submit" class="btn btn-primary">Load Data</button>
                </form>
                
                <?php if (!empty($sheetNames)): ?>
                <!-- Non-form sheet selector for quick switching without form submission -->
                <div class="mt-3">
                    <label class="form-label">Quick Sheet Switch:</label>
                    <div class="d-flex">
                        <?php foreach ($sheetNames as $sheet): ?>
                            <a href="index.php?id=<?php echo urlencode($spreadsheetId); ?>&sheet=<?php echo urlencode($sheet); ?>"
                               class="btn btn-sm <?php echo ($sheet === $sheetName) ? 'btn-secondary' : 'btn-outline-secondary'; ?> me-2">
                                <?php echo htmlspecialchars($sheet); ?>
                            </a>
                        <?php endforeach; ?>
                    </div>
                </div>
                <?php endif; ?>
                
                <?php if (!empty($error)): ?>
                <div class="error mt-3">
                    <strong>Error:</strong> <?php echo htmlspecialchars($error); ?>
                </div>
                <?php endif; ?>
            </div>
        </div>
        
        <?php if (!empty($data)): ?>
        <div class="card">
            <div class="card-header">
                <h5>Sheet Data: <?php echo htmlspecialchars($sheetName); ?></h5>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-striped table-bordered">
                        <?php if (!empty($data)): ?>
                            <thead>
                                <tr>
                                    <?php foreach ($data[0] as $index => $header): ?>
                                    <th><?php echo htmlspecialchars($header); ?></th>
                                    <?php endforeach; ?>
                                </tr>
                            </thead>
                            <tbody>
                                <?php for ($i = 1; $i < count($data); $i++): ?>
                                <tr>
                                    <?php foreach ($data[$i] as $cell): ?>
                                    <td><?php echo htmlspecialchars($cell); ?></td>
                                    <?php endforeach; ?>
                                </tr>
                                <?php endfor; ?>
                            </tbody>
                        <?php endif; ?>
                    </table>
                </div>
            </div>
        </div>
        <?php endif; ?>
        
        <div class="mt-4">
            <div class="row">
                <div class="col-md-6">
                    <div class="card mb-3">
                        <div class="card-header">
                            <h5>Essential Features</h5>
                        </div>
                        <div class="card-body">
                            <ul class="list-group">
                                <li class="list-group-item"><a href="real_login.php" class="fw-bold text-success">Real Google Sheets Login</a> - Authenticate with actual Google Sheets data</li>
                                <li class="list-group-item"><a href="sheets_auth.php">Test User Login</a> - Test authentication using Google Sheets</li>
                                <li class="list-group-item"><a href="user_example.php">User-Specific Data</a> - View filtered data based on user</li>
                            </ul>
                        </div>
                    </div>
                </div>
                <div class="col-md-6">
                    <div class="card mb-3">
                        <div class="card-header">
                            <h5>Documentation</h5>
                        </div>
                        <div class="card-body">
                            <ul class="list-group">
                                <li class="list-group-item"><a href="instructions/" target="_blank">Google Sheets Setup Guide</a></li>
                                <li class="list-group-item"><a href="sample_users.csv" download>Download Users CSV Template</a></li>
                                <li class="list-group-item"><a href="sample_user_data.csv" download>Download User Data CSV Template</a></li>
                            </ul>
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="card mt-3">
                <div class="card-header">
                    <h5>Sheet Setup Instructions</h5>
                </div>
                <div class="card-body">
                    <ol>
                        <li>Create two sheets in your Google Spreadsheet:
                            <ul>
                                <li><strong>users</strong> - For user authentication data</li>
                                <li><strong>user_data</strong> - For user-specific data</li>
                            </ul>
                        </li>
                        <li>Download the CSV templates above and import them into the corresponding sheets</li>
                        <li>Make sure to share your Google Sheet with your service account email</li>
                        <li>Use the "Load Data" button above to view your sheet data</li>
                        <li>Try the "Test User Login" feature to test authentication</li>
                    </ol>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0-alpha1/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
