<?php
// Initialize variables
$message = '';
$error = '';
$setupComplete = false;

// Check if form is submitted
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['json_credentials'])) {
    $jsonCredentials = trim($_POST['json_credentials']);
    $spreadsheetId = isset($_POST['spreadsheet_id']) ? trim($_POST['spreadsheet_id']) : '';
    
    // Validate JSON
    $credentials = json_decode($jsonCredentials, true);
    if ($credentials === null) {
        $error = "Invalid JSON format. Please check your credentials.";
    } else {
        // Check for required fields
        $requiredFields = ['type', 'project_id', 'private_key_id', 'private_key', 'client_email', 'client_id'];
        $missingFields = [];
        
        foreach ($requiredFields as $field) {
            if (!isset($credentials[$field]) || empty($credentials[$field])) {
                $missingFields[] = $field;
            }
        }
        
        if (!empty($missingFields)) {
            $error = "Missing required fields in JSON: " . implode(', ', $missingFields);
        } else {
            // All validations passed, proceed with setup
            
            // 1. Update config.php with the values from the JSON
            $configTemplate = file_get_contents('config.php');
            
            // Replace values in config.php
            $configTemplate = preg_replace(
                "/define\('API_KEY', '(.*)'\);/",
                "define('API_KEY', '');",
                $configTemplate
            );
            
            $configTemplate = preg_replace(
                "/define\('CLIENT_ID', (.*)\);/",
                "define('CLIENT_ID', '" . addslashes($credentials['client_id']) . "');",
                $configTemplate
            );
            
            $configTemplate = preg_replace(
                "/define\('CLIENT_EMAIL', (.*)\);/",
                "define('CLIENT_EMAIL', '" . addslashes($credentials['client_email']) . "');",
                $configTemplate
            );
            
            $configTemplate = preg_replace(
                "/define\('PRIVATE_KEY', (.*)\);/",
                "define('PRIVATE_KEY', '" . addslashes($credentials['private_key']) . "');",
                $configTemplate
            );
            
            if (!empty($spreadsheetId)) {
                $configTemplate = preg_replace(
                    "/define\('SPREADSHEET_ID', '(.*)'\);/",
                    "define('SPREADSHEET_ID', '" . addslashes($spreadsheetId) . "');",
                    $configTemplate
                );
            }
            
            // Save updated config
            file_put_contents('config.php', $configTemplate);
            
            // 2. Empty credentials.json for security (don't store sensitive data in JSON file)
            $emptyCredentials = '{
  "type": "service_account",
  "project_id": "",
  "private_key_id": "",
  "private_key": "",
  "client_email": "",
  "client_id": "",
  "auth_uri": "https://accounts.google.com/o/oauth2/auth",
  "token_uri": "https://oauth2.googleapis.com/token",
  "auth_provider_x509_cert_url": "https://www.googleapis.com/oauth2/v1/certs",
  "client_x509_cert_url": "",
  "universe_domain": "googleapis.com"
}';
            file_put_contents('credentials.json', $emptyCredentials);
            
            // Setup complete
            $setupComplete = true;
            $message = "Setup completed successfully! Your Google Sheets API credentials have been configured.";
        }
    }
}

// Check if setup is already complete by looking at config.php
$alreadySetup = false;
require_once 'config.php';

// Only consider setup complete if config.php contains real values, not placeholders
if (defined('CLIENT_ID') && !empty(CLIENT_ID) && CLIENT_ID !== 'your-client-id' &&
    defined('CLIENT_EMAIL') && !empty(CLIENT_EMAIL) && CLIENT_EMAIL !== 'your-service-account-email' &&
    defined('PRIVATE_KEY') && !empty(PRIVATE_KEY) && PRIVATE_KEY !== 'your-private-key') {
    $alreadySetup = true;
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Google Sheets API Setup</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0-alpha1/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        body {
            padding: 20px;
            background-color: #f8f9fa;
        }
        .container {
            max-width: 800px;
        }
        .card {
            margin-bottom: 20px;
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
        }
        .setup-form {
            margin-top: 20px;
        }
        .json-input {
            font-family: monospace;
            min-height: 200px;
        }
        .step {
            margin-bottom: 30px;
        }
        .step-number {
            display: inline-block;
            width: 30px;
            height: 30px;
            background-color: #0d6efd;
            color: white;
            border-radius: 50%;
            text-align: center;
            line-height: 30px;
            margin-right: 10px;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1 class="my-4 text-center">Google Sheets API Setup</h1>
        
        <?php if (!empty($error)): ?>
        <div class="alert alert-danger"><?php echo htmlspecialchars($error); ?></div>
        <?php endif; ?>
        
        <?php if (!empty($message)): ?>
        <div class="alert alert-success"><?php echo htmlspecialchars($message); ?></div>
        <?php endif; ?>
        
        <?php if ($setupComplete || $alreadySetup): ?>
            <div class="card">
                <div class="card-header bg-success text-white">
                    <h5 class="mb-0">Setup Complete</h5>
                </div>
                <div class="card-body">
                    <p>Your Google Sheets API credentials have been configured successfully.</p>
                    <p>Next steps:</p>
                    <ol>
                        <li>Create two sheets in your Google Spreadsheet:
                            <ul>
                                <li><strong>users</strong> - For user authentication data</li>
                                <li><strong>user_data</strong> - For user-specific data</li>
                            </ul>
                        </li>
                        <li>Download and import the CSV templates for each sheet</li>
                        <li>Make sure your Google Sheet is shared with your service account email with <strong>Editor</strong> permissions</li>
                        <li><strong>Important:</strong> The "user_id" column in the "user_data" sheet must match the numeric "id" values in the "users" sheet (e.g., 1, 2, 3) for proper data filtering</li>
                    </ol>
                    
                    <?php if ($setupComplete): ?>
                    <div class="alert alert-info">
                        <strong>Note:</strong> For security reasons, your credentials have been securely stored in the application configuration and removed from the credentials.json file.
                    </div>
                    <?php endif; ?>
                    
                    <a href="index.php" class="btn btn-primary">Go to Main Application</a>
                    <a href="instructions/" class="btn btn-info ms-2">View Setup Guide</a>
                </div>
            </div>
        <?php else: ?>
            <div class="card">
                <div class="card-header">
                    <h5 class="mb-0">Setup Instructions</h5>
                </div>
                <div class="card-body">
                    <div class="step">
                        <h4><span class="step-number">1</span>Create a Google Cloud Project</h4>
                        <p>If you haven't already, create a project in the <a href="https://console.cloud.google.com/" target="_blank">Google Cloud Console</a>.</p>
                    </div>
                    
                    <div class="step">
                        <h4><span class="step-number">2</span>Enable the Google Sheets API</h4>
                        <p>In your Google Cloud project, navigate to "APIs & Services" > "Library" and enable the Google Sheets API.</p>
                    </div>
                    
                    <div class="step">
                        <h4><span class="step-number">3</span>Create a Service Account</h4>
                        <p>Go to "APIs & Services" > "Credentials" and create a new service account.</p>
                        <p>Give it a name and grant it the necessary permissions (Project > <strong>Editor</strong> role is recommended to allow both reading and writing to sheets).</p>
                    </div>
                    
                    <div class="step">
                        <h4><span class="step-number">4</span>Create a Service Account Key</h4>
                        <p>For your service account, create a new key (JSON format).</p>
                        <p>This will download a JSON file containing your credentials.</p>
                    </div>
                    
                    <div class="step">
                        <h4><span class="step-number">5</span>Create a Google Sheet</h4>
                        <p>Create a new Google Sheet or use an existing one.</p>
                        <p>Share this sheet with the service account email address (with <strong>Editor</strong> permissions to allow both reading and writing to the sheet).</p>
                        <p>Note the Spreadsheet ID from the URL: <code>https://docs.google.com/spreadsheets/d/<strong>spreadsheet-id</strong>/edit</code></p>
                    </div>
                    
                    <div class="step">
                        <h4><span class="step-number">6</span>Enter Your Credentials Below</h4>
                        <p>Copy and paste the entire contents of your downloaded JSON credentials file.</p>
                    </div>
                </div>
            </div>
            
            <div class="card">
                <div class="card-header">
                    <h5 class="mb-0">Enter Your Credentials</h5>
                </div>
                <div class="card-body">
                    <form method="post" action="" class="setup-form">
                        <div class="mb-3">
                            <label for="json_credentials" class="form-label">Service Account JSON Credentials</label>
                            <textarea class="form-control json-input" id="json_credentials" name="json_credentials" rows="10" required></textarea>
                            <div class="form-text">
                                Paste the entire contents of your service account JSON file here.
                                This information will only be stored locally in your application and will not be sent anywhere else.
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label for="spreadsheet_id" class="form-label">Google Spreadsheet ID (Optional)</label>
                            <input type="text" class="form-control" id="spreadsheet_id" name="spreadsheet_id">
                            <div class="form-text">
                                The Spreadsheet ID is the value between /d/ and /edit in the Google Sheet URL.<br>
                                Example: https://docs.google.com/spreadsheets/d/<strong>1BxiMVs0XRA5nFMdKvBdBZjgmUUqptlbs74OgvE2upms</strong>/edit
                            </div>
                        </div>
                        
                        <div class="alert alert-info">
                            <strong>Security Note:</strong> Your credentials will only be stored locally in this application.
                            After setup, the JSON data will be securely stored in the application's configuration.
                        </div>
                        
                        <button type="submit" class="btn btn-primary">Complete Setup</button>
                    </form>
                </div>
            </div>
        <?php endif; ?>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0-alpha1/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>