<?php
require_once 'config.php';

/**
 * Generate a JWT (JSON Web Token) for Google API authentication
 * 
 * @return string The JWT token
 */
function generateJWT() {
    $header = [
        'alg' => 'RS256',
        'typ' => 'JWT'
    ];
    
    $time = time();
    $claim = [
        'iss' => CLIENT_EMAIL,
        'scope' => 'https://www.googleapis.com/auth/spreadsheets.readonly',
        'aud' => 'https://oauth2.googleapis.com/token',
        'exp' => $time + 3600,
        'iat' => $time
    ];
    
    $base64Header = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode(json_encode($header)));
    $base64Claim = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode(json_encode($claim)));
    
    $signature = '';
    openssl_sign(
        $base64Header . '.' . $base64Claim,
        $signature,
        PRIVATE_KEY,
        'SHA256'
    );
    
    $base64Signature = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($signature));
    
    return $base64Header . '.' . $base64Claim . '.' . $base64Signature;
}

/**
 * Get an access token from Google OAuth
 * 
 * @return string The access token
 */
function getAccessToken() {
    $jwt = generateJWT();
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, 'https://oauth2.googleapis.com/token');
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query([
        'grant_type' => 'urn:ietf:params:oauth:grant-type:jwt-bearer',
        'assertion' => $jwt
    ]));
    
    $response = curl_exec($ch);
    curl_close($ch);
    
    $data = json_decode($response, true);
    
    if (isset($data['access_token'])) {
        return $data['access_token'];
    } else {
        throw new Exception('Failed to get access token: ' . print_r($data, true));
    }
}

/**
 * Get data from Google Sheets
 * 
 * @param string $spreadsheetId The ID of the spreadsheet
 * @param string $range The range to fetch (e.g., 'Sheet1!A1:D10')
 * @return array The data from the spreadsheet
 */
function getSheetData($spreadsheetId, $range) {
    $accessToken = getAccessToken();
    
    $url = "https://sheets.googleapis.com/v4/spreadsheets/{$spreadsheetId}/values/{$range}";
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Authorization: Bearer ' . $accessToken
    ]);
    
    $response = curl_exec($ch);
    curl_close($ch);
    
    $data = json_decode($response, true);
    
    if (isset($data['values'])) {
        return $data['values'];
    } else {
        throw new Exception('Failed to get sheet data: ' . print_r($data, true));
    }
}

/**
 * Get spreadsheet metadata
 * 
 * @param string $spreadsheetId The ID of the spreadsheet
 * @return array The spreadsheet metadata
 */
function getSpreadsheetMetadata($spreadsheetId) {
    $accessToken = getAccessToken();
    
    $url = "https://sheets.googleapis.com/v4/spreadsheets/{$spreadsheetId}";
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Authorization: Bearer ' . $accessToken
    ]);
    
    $response = curl_exec($ch);
    curl_close($ch);
    
    return json_decode($response, true);
}

/**
 * Get all sheet names from a spreadsheet
 * 
 * @param string $spreadsheetId The ID of the spreadsheet
 * @return array The sheet names
 */
function getSheetNames($spreadsheetId) {
    $metadata = getSpreadsheetMetadata($spreadsheetId);
    $sheets = [];
    
    if (isset($metadata['sheets'])) {
        foreach ($metadata['sheets'] as $sheet) {
            $sheets[] = $sheet['properties']['title'];
        }
    }
    
    return $sheets;
}
?>