<?php
session_start();
require_once 'config.php';
require_once 'sheets_api.php';

$error = '';
$message = '';
$isLoggedIn = isset($_SESSION['user_id']);
$userData = $isLoggedIn ? [
    'id' => $_SESSION['user_id'],
    'email' => $_SESSION['user_email'],
    'name' => $_SESSION['user_name'],
    'role' => $_SESSION['user_role']
] : null;

// Handle login
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['login'])) {
    $email = isset($_POST['email']) ? trim($_POST['email']) : '';
    $password = isset($_POST['password']) ? $_POST['password'] : '';
    
    if (empty($email) || empty($password)) {
        $error = "Please enter both email and password";
    } else {
        try {
            // Get spreadsheet ID from config
            $spreadsheetId = defined('SPREADSHEET_ID') && !empty(SPREADSHEET_ID) ? SPREADSHEET_ID : '';
            
            if (empty($spreadsheetId)) {
                $error = "Spreadsheet ID is not configured. Please set it in config.php";
            } else {
                // Get user data from the "users" sheet
                $sheetName = 'users';
                $userData = authenticateUser($spreadsheetId, $sheetName, $email, $password);
                
                if ($userData) {
                    // Set session variables
                    $_SESSION['user_id'] = $userData['id'];
                    $_SESSION['user_email'] = $userData['email'];
                    $_SESSION['user_name'] = $userData['name'];
                    $_SESSION['user_role'] = $userData['role'];
                    
                    // Update last login time (this would require write access to the sheet)
                    // updateLastLogin($spreadsheetId, $sheetName, $userData['id']);
                    
                    // Set logged in status in session
                    $isLoggedIn = true;
                    
                    // Redirect to avoid form resubmission
                    header('Location: ' . $_SERVER['PHP_SELF'] . '?login_success=1');
                    exit;
                } else {
                    $error = "Invalid email or password";
                }
            }
        } catch (Exception $e) {
            $error = "Error: " . $e->getMessage();
        }
    }
}

// Handle success message after redirect
if (isset($_GET['login_success'])) {
    $message = "Login successful! Welcome, " . $_SESSION['user_name'];
}

// Handle logout
if (isset($_GET['logout'])) {
    session_destroy();
    $_SESSION = [];
    $message = "You have been logged out.";
    $isLoggedIn = false;
    
    // Redirect to remove the logout parameter
    header("Location: " . $_SERVER['PHP_SELF']);
    exit;
}

// Check if user is already logged in
if (isset($_SESSION['user_id'])) {
    $isLoggedIn = true;
    $userData = [
        'id' => $_SESSION['user_id'],
        'email' => $_SESSION['user_email'],
        'name' => $_SESSION['user_name'],
        'role' => $_SESSION['user_role']
    ];
}

/**
 * Authenticate user against Google Sheets data
 * 
 * @param string $spreadsheetId The ID of the spreadsheet
 * @param string $sheetName The name of the sheet containing user data
 * @param string $email The user's email
 * @param string $password The user's password
 * @return array|false User data if authenticated, false otherwise
 */
function authenticateUser($spreadsheetId, $sheetName, $email, $password) {
    try {
        // Get all data from the sheet
        $allData = getSheetData($spreadsheetId, $sheetName);
        
        if (empty($allData) || count($allData) < 2) {
            // No data or only headers
            return false;
        }
        
        // Get column indexes
        $headers = $allData[0];
        $emailIndex = array_search('email', $headers);
        $passwordIndex = array_search('password', $headers);
        $idIndex = array_search('id', $headers);
        $nameIndex = array_search('name', $headers);
        $roleIndex = array_search('role', $headers);
        $statusIndex = array_search('status', $headers);
        
        // Check if required columns exist
        if ($emailIndex === false || $passwordIndex === false) {
            throw new Exception("Required columns 'email' and 'password' not found in the sheet");
        }
        
        // Search for matching email and password
        for ($i = 1; $i < count($allData); $i++) {
            $row = $allData[$i];
            
            // Check if the row has enough columns
            if (count($row) <= max($emailIndex, $passwordIndex)) {
                continue;
            }
            
            // Check if email and password match
            if ($row[$emailIndex] === $email && $row[$passwordIndex] === $password) {
                // Check if account is active (if status column exists)
                if ($statusIndex !== false && isset($row[$statusIndex]) && $row[$statusIndex] !== 'active') {
                    return false; // Account is not active
                }
                
                // Return user data
                return [
                    'id' => $idIndex !== false && isset($row[$idIndex]) ? $row[$idIndex] : $i,
                    'email' => $row[$emailIndex],
                    'name' => $nameIndex !== false && isset($row[$nameIndex]) ? $row[$nameIndex] : 'User',
                    'role' => $roleIndex !== false && isset($row[$roleIndex]) ? $row[$roleIndex] : 'user'
                ];
            }
        }
        
        // No matching user found
        return false;
    } catch (Exception $e) {
        throw $e;
    }
}

/**
 * Update the last login time for a user
 * Note: This would require write access to the sheet
 * 
 * @param string $spreadsheetId The ID of the spreadsheet
 * @param string $sheetName The name of the sheet containing user data
 * @param string $userId The user's ID
 * @return bool Success or failure
 */
function updateLastLogin($spreadsheetId, $sheetName, $userId) {
    // This is a placeholder function
    // Implementing this would require using the Google Sheets API with write access
    // For simplicity, we're not implementing this in this example
    return true;
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Google Sheets Authentication</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0-alpha1/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        body {
            padding: 20px;
            background-color: #f8f9fa;
        }
        .container {
            max-width: 800px;
        }
        .card {
            margin-bottom: 20px;
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
        }
        .login-form {
            max-width: 400px;
            margin: 0 auto;
        }
        .user-info {
            background-color: #e9ecef;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1 class="my-4 text-center">Google Sheets Authentication</h1>
        
        <?php if (!empty($error)): ?>
        <div class="alert alert-danger"><?php echo htmlspecialchars($error); ?></div>
        <?php endif; ?>
        
        <?php if (!empty($message)): ?>
        <div class="alert alert-success"><?php echo htmlspecialchars($message); ?></div>
        <?php endif; ?>
        
        <?php if ($isLoggedIn): ?>
            <!-- User is logged in, show dashboard -->
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5 class="mb-0">User Dashboard</h5>
                    <a href="?logout=1" class="btn btn-sm btn-outline-danger">Logout</a>
                </div>
                <div class="card-body">
                    <div class="user-info">
                        <h4>Welcome, <?php echo htmlspecialchars($userData['name']); ?>!</h4>
                        <p><strong>Email:</strong> <?php echo htmlspecialchars($userData['email']); ?></p>
                        <p><strong>Role:</strong> <?php echo htmlspecialchars($userData['role']); ?></p>
                        <p><strong>User ID:</strong> <?php echo htmlspecialchars($userData['id']); ?></p>
                    </div>
                    
                    <h5>What's Next?</h5>
                    <p>Now that you're logged in, you can:</p>
                    <ul>
                        <li>Display user-specific data from Google Sheets</li>
                        <li>Allow users to update their information</li>
                        <li>Show different content based on user roles</li>
                        <li>Create a personalized dashboard</li>
                    </ul>
                    
                    <div class="mt-4">
                        <a href="user_example.php" class="btn btn-primary">View User-Specific Data Example</a>
                    </div>
                </div>
            </div>
        <?php else: ?>
            <!-- User is not logged in, show login form -->
            <div class="card">
                <div class="card-header">
                    <h5 class="mb-0">Login with Google Sheets</h5>
                </div>
                <div class="card-body">
                    <form method="post" action="" class="login-form">
                        <div class="mb-3">
                            <label for="email" class="form-label">Email</label>
                            <input type="email" class="form-control" id="email" name="email" required>
                        </div>
                        <div class="mb-3">
                            <label for="password" class="form-label">Password</label>
                            <input type="password" class="form-control" id="password" name="password" required>
                        </div>
                        <button type="submit" name="login" class="btn btn-primary">Login</button>
                    </form>
                    
                    <div class="mt-4">
                        <h6>Sample Users (from sample_users.csv):</h6>
                        <ul>
                            <li>Email: john.doe@example.com, Password: password123</li>
                            <li>Email: jane.smith@example.com, Password: securepass456</li>
                            <li>Email: bob.johnson@example.com, Password: bobpass789</li>
                        </ul>
                    </div>
                </div>
            </div>
        <?php endif; ?>
        
        <div class="card mt-4">
            <div class="card-header">
                <h5 class="mb-0">Testing User Login</h5>
            </div>
            <div class="card-body">
                <p>This page demonstrates how to authenticate users using Google Sheets as a user database.</p>
                
                <h6 class="mt-3">Sample Users for Testing:</h6>
                <div class="table-responsive">
                    <table class="table table-sm table-bordered">
                        <thead>
                            <tr>
                                <th>Email</th>
                                <th>Password</th>
                                <th>Role</th>
                            </tr>
                        </thead>
                        <tbody>
                            <tr>
                                <td>john.doe@example.com</td>
                                <td>password123</td>
                                <td>user</td>
                            </tr>
                            <tr>
                                <td>jane.smith@example.com</td>
                                <td>securepass456</td>
                                <td>admin</td>
                            </tr>
                            <tr>
                                <td>bob.johnson@example.com</td>
                                <td>bobpass789</td>
                                <td>user</td>
                            </tr>
                        </tbody>
                    </table>
                </div>
                
                <div class="alert alert-info mt-3">
                    <strong>Note:</strong> Make sure you've imported the sample_users.csv file into a sheet named "users" in your Google Spreadsheet.
                </div>
                
                <a href="index.php" class="btn btn-primary">Back to Main Page</a>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0-alpha1/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>